package WebService::Stripe;
use Moo;
with 'WebService::Client';

our $VERSION = '0.0400'; # VERSION

use Carp qw(croak);
use Method::Signatures;

has api_key => (
    is       => 'ro',
    required => 1,
);

has version => (
    is      => 'ro',
    default => '2014-11-05',
);

has '+base_url' => ( default => 'https://api.stripe.com' );

has '+content_type' => ( default => 'application/x-www-form-urlencoded' );

method BUILD(@args) {
    $self->ua->default_headers->authorization_basic( $self->api_key, '' );
    $self->ua->default_header( 'Stripe-Version' => '2014-11-05' );
}

method next(HashRef $thing!, HashRef :$query) {
    $query ||= {};
    return undef unless $thing->{has_more};
    my $starting_after = $thing->{data}[-1]{id} or return undef;
    return $self->get( $thing->{url},
        { %$query, starting_after => $starting_after } );
}

method create_customer(HashRef $data={}) {
    return $self->post( "/v1/customers", $data );
}

method get_customer(Str $id) {
    return $self->get( "/v1/customers/$id" );
}

method update_customer(Str $id, HashRef $data) {
    return $self->post( "/v1/customers/$id", $data );
}

method get_customers(HashRef :$query) {
    return $self->get( "/v1/customers", $query );
}

method create_card(HashRef $data, :$customer_id!) {
    return $self->post( "/v1/customers/$customer_id/cards", $data );
}

method get_charge(Str $id) {
    return $self->get( "/v1/charges/$id" );
}

method create_charge(HashRef $data) {
    return $self->post( "/v1/charges", $data );
}

method capture_charge(Str $id, HashRef :$data) {
    return $self->post( "/v1/charges/$id/capture", $data );
}

method refund_charge(Str $id, HashRef :$data) {
    return $self->post( "/v1/charges/$id/refunds", $data );
}

method create_token(HashRef $data) {
    return $self->post( "/v1/tokens", $data );
}

method get_token(Str $id) {
    return $self->get( "/v1/tokens/$id" );
}

method create_account(HashRef $data) {
    return $self->post( "/v1/accounts", $data );
}

method get_account(Str $id) {
    return $self->get( "/v1/accounts/$id" );
}

method update_account(Str $id, HashRef :$data) {
    return $self->post( "/v1/accounts/$id", $data );
}

method add_bank(HashRef $data, Str :$account_id) {
    return $self->post( "/v1/accounts/$account_id/bank_accounts", $data );
}

# ABSTRACT: Stripe API bindings


1;

__END__

=pod

=encoding UTF-8

=head1 NAME

WebService::Stripe - Stripe API bindings

=head1 VERSION

version 0.0400

=head1 SYNOPSIS

    my $stripe = WebService::Stripe->new(
        api_key => 'secret',
        version => '2014-11-05', # optional
    );
    my $customer = $stripe->get_customer('cus_57eDUiS93cycyH');

=head1 METHODS

=head2 get_customer

    get_customer($id)

Returns the customer for the given id.

=head2 create_customer

    create_customer($data)

Creates a customer.
The C<$data> hashref is optional.
Returns the customer.

Example:

    $customer = $stripe->create_customer({ email => 'bob@foo.com' });

=head2 update_customer

    update_customer($id, $data)

Updates a customer.
Returns the updated customer.

Example:

    $customer = $stripe->update_customer($id, { description => 'foo' });

=head2 get_customers

    get_customers(query => $query)

Returns a list of customers.
The query param is optional.

=head2 next

    next($collection)

Returns the next page of results for the given collection.

Example:

    my $customers = $s->get_customers;
    ...
    while ($customers = $s->next($customers)) {
        ...
    }

=head2 create_card

    create_card($data, customer_id => 'cus_123')

=head2 get_charge

    get_charge($id)

Returns the charge for the given id.

=head2 create_charge

    create_charge($data)

Creates a charge.

=head2 capture_charge

    capture_charge($id, data => $data)

Captures the charge with the given id.
The data param is optional.

=head2 refund_charge

    refund_charge($id, data => $data)

Refunds the charge with the given id.
The data param is optional.

=head2 get_token

    get_token($id)

=head2 create_token

    create_token($data)

=head2 get_account

    get_account($id)

=head2 create_account

    create_account($data)

=head2 update_account

    update_account($id, data => $data)

=head2 add_bank

    add_bank($data, account_id => $account_id)

Add a new bank account.

Example:

    my $account = $stripe->create_account({
        managed => 'true',
        country => 'CA',
    });

    my $bank = $s->add_bank(
        {
            'bank_account[country]'        => 'CA',
            'bank_account[currency]'       => 'cad',
            'bank_account[routing_number]' => '00022-001',
            'bank_account[account_number]' => '000123456789',
        },
        account_id => $account->{id},
    );

=head1 AUTHOR

Naveed Massjouni <naveed@vt.edu>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Tilt, Inc..

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
