#!/bin/ksh

#############################################################################
#
# NcFTPd General Report Generator
# ===============================
#
# Copyright 1998, Mike Gleason.  All rights reserved.
#
# Function:
#
#   *  This script is responsible for generating the "General" report,
#      in HTML output.
#
# Requirements:
#
#   *  gnuplot
#
#   *  awk
#
#   *  datemunger
#
#############################################################################

. /usr/local/etc/ncftpd/rptbin/ncftpd_reports.env

#############################################################################
# No more user-configurable options below this point.
#############################################################################

whoiam1=`$WHOAMI 2>/dev/null`
if [ "$whoiam1" = "" ] ; then whoiam="web" ; else whoiam="<b>$whoiam1</b>" ; fi

onhost1=`uname -n 2>/dev/null`
if [ "$onhost1" != "" ] ; then onhost=" on <b>$onhost1</b>" ; else onhost="" ; fi

#############################################################################
# Start the HTML page
#############################################################################

echo "Content-type: text/html"
echo ""
echo "<html>"
echo "<head>"
echo "<title>NcFTPd General Report</title>"
echo "</head>"
echo "<body>"
echo "<!-- user: $whoiam1  host: $onhost1 -->"

#############################################################################
# Setup the temporary directory
#############################################################################

nowpfx=`$DATEMUNGER now formatted "%Y%m%d.%H%M."`
otdir="$ODIR/$nowpfx$$"
relotdir="$RELODIR/$nowpfx$$"

umask 022
if [ ! -d "$ODIR" ] ; then mkdir "$ODIR" 2>/dev/null ; fi
if [ ! -d "$otdir" ] ; then
	mkdir "$otdir" 2>/dev/null
	if [ $? -ne 0 ] ; then
		echo "<p>Could not create the temporary output directory <TT>$otdir</TT>."
		echo "<p><address></address></body></html>"
		exit 2
	fi
fi
# csh -c "( sleep 300 ; cd $PWD ; rm -rf $otdir ) &"

#############################################################################
# Sanity checks
#############################################################################

if [ ! -x "$SUM_STAT_LOGS" ] ; then
	echo "<p>Cannot execute <TT>$SUM_STAT_LOGS</TT> as $whoiam user${onhost}."
	echo "<p><address></address></body></html>"
	exit 2
fi

#############################################################################
# Validate the input
#############################################################################

if [ "$#" -ne 10 ] ; then
	echo "<p>Not enough parameters ($# of 10) given to generate report.</p>"
	echo "<p><address></address></body></html>"
	exit 2
fi

start_Y="$1"
start_B="$2"
start_d="$3"
start_H="$4"
start_M="$5"

shift 5
end_Y="$1"
end_B="$2"
end_d="$3"
end_H="$4"
end_M="$5"


##############################################################################
# Convert the "%B" form of the month into "%m" form
# (Unfortunate english language dependency)
##############################################################################

B="$start_B"
case "$B" in
	Jan*)	m=01 ;; Feb*)	m=02 ;; Mar*)	m=03 ;;
	Apr*)	m=04 ;; May*)	m=05 ;; Jun*)	m=06 ;;
	Jul*)	m=07 ;; Aug*)	m=08 ;; Sep*)	m=09 ;;
	Oct*)	m=10 ;; Nov*)	m=11 ;; Dec*)	m=12 ;;
	[0-9]*)	m="$B";; *)	m=01 ;;
esac
start_m="$m"

B="$end_B"
case "$B" in
	Jan*)	m=01 ;; Feb*)	m=02 ;; Mar*)	m=03 ;;
	Apr*)	m=04 ;; May*)	m=05 ;; Jun*)	m=06 ;;
	Jul*)	m=07 ;; Aug*)	m=08 ;; Sep*)	m=09 ;;
	Oct*)	m=10 ;; Nov*)	m=11 ;; Dec*)	m=12 ;;
	[0-9]*)	m="$B";; *)	m=01 ;;
esac
end_m="$m"

if [ ! -x "$STAT_LOG_DIR" ] ; then
	echo "<p>Error accessing the report data directory${onhost}:</p>"
	echo "<p>"
	ls -l "$STAT_LOG_DIR" 2>&1	# print an error
	echo "<p>Make sure the $whoiam user has read and execute permission on <tt>$STAT_LOG_DIR</tt>, and execute permission on the parent directories."
	echo "<p><address></address></body></html>"
	exit 2
fi
cd "$STAT_LOG_DIR"

##############################################################################
# Convert the dates into timestamps.
# This is needed so we can compute the names of the monthly log files
# that will be used to get the stats from.
##############################################################################

start_date="${start_Y}-${start_m}-${start_d} ${start_H}:${start_M}:00"
start_ts=`$DATEMUNGER "$start_date"`
case "$start_ts" in
	[0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9]*)
		;;
	*)
		echo "<p>Could not convert \"$start_date\" back to a timestamp."
		echo "<p><address></address></body></html>"
		exit 1
esac

end_date="${end_Y}-${end_m}-${end_d} ${end_H}:${end_M}:00"
end_ts=`$DATEMUNGER "$end_date"`
case "$end_ts" in
	[0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9]*)
		;;
	*)
		echo "<p>Could not convert \"$end_date\" back to a timestamp."
		echo "<p><address></address></body></html>"
		exit 1
esac

##############################################################################
# Compute the lognames.
##############################################################################

start_yyyymm=`$DATEMUNGER "$start_ts" formatted "%Y%m"`
start_yyyymmddHHMMSS=`$DATEMUNGER "$start_ts" formatted "%Y%m%d%H%M%S"`
end_yyyymm=`$DATEMUNGER "$end_ts" formatted "%Y%m"`
end_yyyymmddHHMMSS=`$DATEMUNGER "$end_ts" formatted "%Y%m%d%H%M%S"`

echo "<!-- timestamps:  start=$start_ts             end=$end_ts -->"
echo "<!-- months:      start=$start_yyyymm         end=$end_yyyymm -->"
echo "<!-- full:        start=$start_yyyymmddHHMMSS end=$end_yyyymmddHHMMSS-->"

yyyymm="$start_yyyymm"
ts="$start_ts"
statlogs=""
nstatlogs=0
while [ "$yyyymm" -le "$end_yyyymm" ] ; do
	nstatlogs=`expr "$nstatlogs" + 1`
	if [ "$nstatlogs" -gt 100 ] ; then break ; fi
	statlog=`$DATEMUNGER "$ts" formatted "$STAT_LOG_SPEC"`
	if [ -r "$statlog" ] ; then
		if [ "$statlogs" = "" ] ; then
			statlogs="$statlog"
		else
			statlogs="$statlogs $statlog"
		fi
	fi
	ts=`$DATEMUNGER "$ts" + 1 mon`
	yyyymm=`$DATEMUNGER "$ts" formatted "%Y%m"`
done

if [ "$statlogs" = "" ] ; then
	echo "<p>No readable stat logs could be found for the requested date range (<i>$start_date</i> to <i>$end_date</i>)."
	echo "<menu><li>Does the directory <tt>$STAT_LOG_DIR</tt> exist${onhost}?</li>"
	echo "<li>Do the stat log files exist in the directory and have names of the form <tt>$STAT_LOG_SPEC</tt>?</li>"
	echo "<li>Do the stat log files have permissions that are accessible by the $whoiam user?</li>"
	echo "<li>Does the directory have permissions accessible by the $whoiam user?</li>"
	echo "<li>Do the parent directories have permissions accessible by the $whoiam user?</li>"
	echo "</menu>"
	echo "<p><address></address></body></html>"
	exit 1
fi

echo "<!-- logfiles: (${statlogs}) -->"
cat $statlogs \
	| "$SUM_STAT_LOGS" "$start_yyyymmddHHMMSS" "$end_yyyymmddHHMMSS" \
	| $AWK -F, -f "$RPTGENERAL_AWK"

##############################################################################
# Optional graphics
##############################################################################

if [ "$PLOTTING_GRAPHS" = yes ] && [ -x "$GNUPLOT" ] ; then
	gpscript="$otdir/gnuplot.script"
	difftimedays=`expr '(' "$end_ts" - "$start_ts" ')' / 86400`
	difftimehours=`expr '(' "$end_ts" - "$start_ts" ')' / 3600`
	difftimesecs=`expr '(' "$end_ts" - "$start_ts" ')'`
	if [ "$difftimedays" -le 3 ] ; then
		intervalsecs=3600
		intervalname="hours"
		if [ "${start_Y}-${start_m}-${start_d}" = "${end_Y}-${end_m}-${end_d}" ] ; then
			xlabel="Hours of ${start_Y}-${start_m}-${start_d}"
			xtics=1
		else
			xlabel="Hours between ${start_Y}-${start_m}-${start_d} and ${end_Y}-${end_m}-${end_d}"
			xtics=4
		fi
	elif [ "$difftimedays" -le 7 ] ; then
		intervalsecs=3600
		intervalname="hours"
		xlabel="Hours between ${start_Y}-${start_m}-${start_d} and ${end_Y}-${end_m}-${end_d}"
		xtics=12
	elif [ "$difftimedays" -le 14 ] ; then
		intervalsecs=3600
		intervalname="hours"
		xlabel="Hours between ${start_Y}-${start_m}-${start_d} and ${end_Y}-${end_m}-${end_d}"
		xtics=24
	elif [ "$difftimedays" -le 125 ] ; then
		intervalsecs=86400
		intervalname="days"
		xlabel="Days between ${start_Y}-${start_m}-${start_d} and ${end_Y}-${end_m}-${end_d}"
		xtics=7
	elif [ "$difftimedays" -le 800 ] ; then
		intervalsecs=604800
		intervalname="weeks"
		xlabel="Weeks between ${start_Y}-${start_m}-${start_d} and ${end_Y}-${end_m}-${end_d}"
		xtics=4
	else
		intervalsecs=2592000
		intervalname="months"
		xlabel="30-day periods between ${start_Y}-${start_m}-${start_d} and ${end_Y}-${end_m}-${end_d}"
		xtics=3
	fi

	intervalmin=`expr "$intervalsecs" / 60`
	intervalright=`expr "$difftimesecs" / "$intervalsecs"`

	intervaldata=$otdir/interval.data
	cat $statlogs | $SUM_STAT_LOGS "$start_yyyymmddHHMMSS" "$end_yyyymmddHHMMSS" "$intervalmin" > $intervaldata

	if [ "$PNG_GRAPH_OUTPUT" = yes ] ; then
		gp_term="png small color"
		gp_fmt="png"
	else
		gp_term="gif small"
		gp_fmt="gif"
	fi

	if [ -s "$intervaldata" ] ; then
		# Create data file to plot
		plotdata="$otdir/gnuplot.data"



		echo "<p><hr><p>"
		#############################################################
		# logins, #
		#############################################################

		name=logins
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"
		title="Number of logins"
		field=8

		$AWK -F, < "$intervaldata" '{
			ts=$2;
			interval = (ts - start_ts) / intervalsecs;
			printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="$field" > $plotdata

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata' title '' with boxes
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# failed logins, #
		#############################################################

		name=failedlogins
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"
		title="Failed logins"

		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="6" > $plotdata.1

		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="7" > $plotdata.2

		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="9" > $plotdata.3

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata.1' title 'refused' with lines, '$plotdata.2' title 'denied' with lines, '$plotdata.3' title 'invalid name/password' with lines

EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript

		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# logins with download, #
		#############################################################

		name=useful_sessions
		title="Useful sessions"
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="13" > $plotdata.1

		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="14" > $plotdata.2

		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="12" > $plotdata.3

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata.1' title 'with a download' with lines, '$plotdata.2' title 'with an upload' with lines, '$plotdata.3' title 'with a directory listing' with lines
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# Simultaneous users
		#############################################################

		name=simul
		title="Simultaneous users"
		field=60
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2;
			interval = (ts - start_ts) / intervalsecs;
			printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="$field" > $plotdata

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata' title '' with boxes
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# modem types 
		#############################################################

		name=modems
		title="Types of user connections"
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $23 + $24 + $25);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="23" > $plotdata.1


		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="26" > $plotdata.2

		$AWK -F, < "$intervaldata" '{
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="27" > $plotdata.3

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata.1' title '33.6/56k/256k modem' with lines, '$plotdata.2' title '10 Mbit' with lines, '$plotdata.3' title '100 Mbit or better' with lines
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		echo "<p><hr><p>"
		#############################################################
		# Downloads, #
		#############################################################

		name=downloads
		title="Number of downloads"
		field=32
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2;
			interval = (ts - start_ts) / intervalsecs;
			printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="$field" > $plotdata

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata' title '' with boxes
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# Downloads, MB
		#############################################################

		name=downloads_MB
		title="Megabytes downloaded"
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2;
			interval = (ts - start_ts) / intervalsecs;

			kb = $41;
			val = kb / 1024.0;

			printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" > $plotdata

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata' title '' with boxes
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# Downloads, throughput
		#############################################################

		name=downloads_avg_thruput
		title="kB/sec download throughput"
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2;
			interval = (ts - start_ts) / intervalsecs;

			kb = $41;
			sec = $45;
			if (sec < 1)
				val = 0;
			else
				val = kb / sec;

			printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" > $plotdata

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata' title '' with boxes
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# download completion %
		#############################################################

		name=dlcompletion
		title="Download error percentage"
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			n = int($32);
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="34" > $plotdata.1

		$AWK -F, < "$intervaldata" '{
			n = int($32);
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="35" > $plotdata.2

		$AWK -F, < "$intervaldata" '{
			n = int($32);
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="36" > $plotdata.3

		$AWK -F, < "$intervaldata" '{
			n = int($32);
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="37" > $plotdata.4

		$AWK -F, < "$intervaldata" '{
			n = int($32);
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="38" > $plotdata.5

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0:100] '$plotdata.1' title 'user abort' with lines, '$plotdata.2' title 'OS error' with lines, '$plotdata.3' title 'incomplete' with lines, '$plotdata.4' title 'no such file' with lines, '$plotdata.5' title 'permission denied' with lines

EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		echo "<p><hr><p>"
		#############################################################
		# Uploads, #
		#############################################################

		name=uploads
		title="Number of uploads"
		field=46
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2;
			interval = (ts - start_ts) / intervalsecs;
			printf("%d %s\n", interval, $field);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="$field" > $plotdata

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata' title '' with boxes
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# Uploads, MB
		#############################################################

		name=uploads_MB
		title="Megabytes uploaded"
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2;
			interval = (ts - start_ts) / intervalsecs;

			kb = $55;
			val = kb / 1024.0;

			printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" > $plotdata

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata' title '' with boxes
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# Uploads, throughput
		#############################################################

		name=uploads_avg_thruput
		title="kB/sec upload throughput"
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			ts=$2;
			interval = (ts - start_ts) / intervalsecs;

			kb = $55;
			sec = $59;
			if (sec < 1)
				val = 0;
			else
				val = kb / sec;

			printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" > $plotdata

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0: ] '$plotdata' title '' with boxes
EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"



		#############################################################
		# upload completion %
		#############################################################

		name=ulcompletion
		title="Upload error percentage"
		gpscript="$otdir/$name.script"
		plotdata="$otdir/$name.data"
		gpresult="$otdir/$name.result"

		$AWK -F, < "$intervaldata" '{
			n = int($46);
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="48" > $plotdata.1

		$AWK -F, < "$intervaldata" '{
			n = int($46);
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="49" > $plotdata.2

		$AWK -F, < "$intervaldata" '{
			n = int($46);
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="50" > $plotdata.3

		$AWK -F, < "$intervaldata" '{
			n = $46;
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="51" > $plotdata.4

		$AWK -F, < "$intervaldata" '{
			n = $46;
			if (n == 0) val = n;
			else val = 100.0 * $field / n;
			ts=$2; interval = (ts - start_ts) / intervalsecs; printf("%d %s\n", interval, val);
		}' start_ts="$start_ts" intervalsecs="$intervalsecs" field="52" > $plotdata.5

		gif=$otdir/$name.$gp_fmt
		relgif=$relotdir/$name.$gp_fmt

		cat > $gpscript <<EOF
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$gif'

			plot [0:$intervalright] [0:100] '$plotdata.1' title 'user abort' with lines, '$plotdata.2' title 'OS error' with lines, '$plotdata.3' title 'incomplete' with lines, '$plotdata.4' title 'no such file' with lines, '$plotdata.5' title 'permission denied' with lines

EOF
		$GNUPLOT > $gpresult 2>&1 < $gpscript
		echo "<p><center><img src=\"$relgif\" alt=\"$title\">"
	fi
else
	echo "<!-- gnuplot is not enabled -->"
fi 

##############################################################################
# End the finished web page.
##############################################################################

echo "<p><address></address></body></html>"

##############################################################################
# Clean up after runs before this one.
##############################################################################

"$CLEANTMPDIR_SH"

##############################################################################
# That's all.
##############################################################################
exit 0
