/*
 *
 * Copyright (C) 2006-2008 Sebastian Trueg <trueg@k3b.org>
 *
 * This file is part of the K3b project.
 * Copyright (C) 1998-2008 Sebastian Trueg <trueg@k3b.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * See the file "COPYING" for the exact licensing terms.
 */

#include "k3bintmapcombobox.h"

#include <KDebug>

#include <QtCore/QHash>
#include <QtCore/QPair>


class K3bIntMapComboBox::Private
{
public:
    QHash<int, int> valueIndexMap;
    QList<QPair<int, QString> > values;

    QString topWhatsThis;
    QString bottomWhatsThis;

    void buildValueIndexMap() {
        valueIndexMap.clear();
        for ( int i = 0; i < values.count(); ++i ) {
            valueIndexMap.insert( values[i].first, i );
        }
    }

    bool haveCustomWhatsThis() const {
        for ( int i = 0; i < values.count(); ++i ) {
            if ( !values[i].second.isEmpty() ) {
                return true;
            }
        }
        return false;
    }

    void updateWhatsThis() {
        if ( haveCustomWhatsThis() ) {
            QString ws( topWhatsThis );
            for( int i = 0; i < values.count(); ++i ) {
                ws += "<p><b>" + q->itemText( i ) + "</b><br>";
                ws += values[i].second;
            }
            ws += "<p>" + bottomWhatsThis;

            q->setWhatsThis( ws );
        }
    }

    K3bIntMapComboBox* q;
};


K3bIntMapComboBox::K3bIntMapComboBox( QWidget* parent )
    : QComboBox( parent ),
      d( new Private() )
{
    d->q = this;

    connect( this, SIGNAL(highlighted(int)),
             this, SLOT(slotItemHighlighted(int)) );
    connect( this, SIGNAL(activated(int)),
             this, SLOT(slotItemActivated(int)) );
}


K3bIntMapComboBox::~K3bIntMapComboBox()
{
    delete d;
}


int K3bIntMapComboBox::selectedValue() const
{
    if( d->values.count() > QComboBox::currentIndex() &&
        QComboBox::currentIndex() >= 0 )
        return d->values[QComboBox::currentIndex()].first;
    else
        return 0;
}


void K3bIntMapComboBox::setSelectedValue( int value )
{
    if( d->valueIndexMap.contains( value ) ) {
        QComboBox::setCurrentIndex( d->valueIndexMap[value] );
    }
}


bool K3bIntMapComboBox::hasValue( int value ) const
{
    return d->valueIndexMap.contains( value );
}


void K3bIntMapComboBox::clear()
{
    d->valueIndexMap.clear();
    d->values.clear();

    QComboBox::clear();
}


bool K3bIntMapComboBox::insertItem( int value, const QString& text, const QString& description, int index )
{
    if( d->valueIndexMap.contains( value ) )
        return false;

    if ( index < 0 || index > QComboBox::count() ) {
        index = QComboBox::count();
    }

    d->values.insert( index, qMakePair<int, QString>( value, description ) );
    d->buildValueIndexMap();

    QComboBox::insertItem( index, text );

    d->updateWhatsThis();

    // select a default value. This is always wanted in K3b
    if ( QComboBox::currentIndex() < 0 ) {
        setSelectedValue( d->values[0].first );
    }

    return true;
}


void K3bIntMapComboBox::slotItemHighlighted( int index )
{
    emit valueHighlighted( d->values[index].first );
}


void K3bIntMapComboBox::slotItemActivated( int index )
{
    emit valueChanged( d->values[index].first );
}


void K3bIntMapComboBox::addGlobalWhatsThisText( const QString& top, const QString& bottom )
{
    d->topWhatsThis = top;
    d->bottomWhatsThis = bottom;
    d->updateWhatsThis();
}

#include "k3bintmapcombobox.moc"
